//==============================================================================
// Project:		Wealth Transfers and their economic effects
// File name:	Figure 2.7 2.9 table B.9 to B.12
// Objective: 	Generate the modelling underpinning figure 2.7 and 2.9 and tables B.9 to B.12
//
//==============================================================================

clear all

************************
*** Data import ***
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

merge m:1 xwaveid wave using bequest_from_HH_death.dta //flag for inheritances due to HH deaths
drop if _merge == 2
drop _merge

merge m:1 wave using CPI.dta //CPI data
drop if _merge == 2
drop _merge

xtset xwaveid wave

***********************************
*** Create equivalisation scale ***
***********************************

gen child = 1 if hgage < 15
gen adult = 1 if hgage >= 15

egen total_child = total(child), by(hhrhid wave)
egen total_adult = total(adult), by(hhrhid wave)

gen equiv_scale = total_child*0.3 + total_adult*0.5 + 0.5 //0.3 for children, 1 for first adult and 0.5 for all subsequent adults

drop if hgint == 0 //drop non-responders

***********************
*** Create variables **
***********************

replace oifinha = 0 if oifinha == -1 //inheritances
replace oifinha = . if oifinha < -1
gen bequest = oifinha
replace bequest = 0 if bequest > 0 & bequest != . & bequest_from_HH_death == 1 //filter out inheritances due to HH deaths

gen bequest_real = bequest * 100 / CPI

gen hgage_squ = hgage^2 //age squared

gen education = . //education
replace education = 1 if edhigh1 == 1 //postgrad
replace education = 2 if edhigh1 == 2 //grad dip or cert
replace education = 3 if edhigh1 == 3 //bachelor or honours
replace education = 4 if edhigh1 == 4 | edhigh1 == 5 //cert (3+) or dip
replace education = 5 if edhigh1 == 8 | edhigh1 == 9 //highschool

gen male = 1 if sex == 1 //gender
gen female = 1 if sex == 2
replace female = 0 if female == .
replace male = 0 if male == .

replace fmmhlq = 0 if fmmhlq == -1 //mother's level of education
replace fmmhlq = . if fmmhlq < -1
gen mother_education_5 = . //first asked in wave 5
replace mother_education_5 = fmmhlq if wave == 5
egen mother_education = min(mother_education_5), by(xwaveid)
replace mother_education = fmmhlq if wave > 5
replace mother_education = 2 if mother_education == 2 | mother_education == 3 | mother_education == 4  //group the VET options
replace mother_education = 3 if mother_education == 5
replace mother_education = 4 if mother_education == 8

replace fmfhlq = 0 if fmfhlq == -1 //father's level of education
replace fmfhlq = . if fmfhlq < -1
gen father_education_5 = . //first asked in wave 5
replace father_education_5 = fmfhlq if wave == 5
egen father_education = min(father_education_5), by(xwaveid)
replace father_education = fmfhlq if wave > 5
replace father_education = 2 if father_education == 2 | father_education == 3 | father_education == 4  //group the VET options
replace father_education = 3 if father_education == 5
replace father_education = 4 if father_education == 8

replace anbcob = . if anbcob < 0 //country of birth
gen cob = anbcob

gen living_with_parents_at_14 = . //living with parents at 14
replace living_with_parents_at_14 = 1 if fmlwop == 1 //both parents
replace living_with_parents_at_14 = 2 if fmlwop == 2 | fmlwop == 3 //one parent and one step parent 
replace living_with_parents_at_14 = 3 if fmlwop == 4 | fmlwop == 5 //single parent
replace living_with_parents_at_14 = 4 if fmlwop == 8 //other

replace tcr = . if tcr < 0 //number of resident children
gen resident_children = tcr

replace bsn = 0 if bsn == -1 //number of siblings
replace bsn = . if bsn < -1
egen siblings = max(bsn), by(xwaveid) //only asked in some waves - imputed for other waves

gen partner = 0 //partner
replace partner = 1 if hhpxid > 0 & hhpxid != .

gen net_worth = hwnwip - hwnwin //household net worth
gen delta_net_worth_4 = net_worth - L4.net_worth
gen delta_net_worth_8 = net_worth - L8.net_worth
gen net_worth_squ = net_worth^2

gen net_worth_real = net_worth * 100 / CPI
gen delta_net_worth_4_real = net_worth_real - L4.net_worth_real
gen delta_net_worth_8_real = net_worth_real - L8.net_worth_real
gen net_worth_squ_real = net_worth_real^2

gen net_worth_e = net_worth / equiv_scale //equivalised net worth

gen HH_disposable_income = hifdip - hifdin //household disposable income

gen L6_4_bequest = L6.bequest + L5.bequest + L4.bequest //pooled inheritances
gen L2_0_bequest = L2.bequest + L.bequest + bequest

gen L6_4_bequest_real = L6.bequest_real + L5.bequest_real + L4.bequest_real //pooled inheritances
gen L2_0_bequest_real = L2.bequest_real + L.bequest_real + bequest_real

***********************
*** Create weights ****
***********************

local weights_4 wlrb_g wlrf_k wlrj_o wlrn_s //balanced longitudinal weights weights from t-4 to t+1  

foreach x of local weights_4{
    egen total_`x' = total(`x'), by(wave)
	egen max_total_`x' = max(total_`x')
	gen `x'_normalised = 0 //need to normalise weights so that earlier and later years have equal weighting
	replace `x'_normalised = `x'/max_total_`x' if `x' > 0
}

gen weight_4 = 0 //all weights in single variable
replace weight_4 = wlrb_g_normalised if wave == 6
replace weight_4 = wlrf_k_normalised if wave == 10
replace weight_4 = wlrj_o_normalised if wave == 14
replace weight_4 = wlrn_s_normalised if wave == 18

local weights_8 wlrb_k wlrf_o wlrj_s //balanced longitudinal weights weights from t-8 to t+1 

foreach x of local weights_8{
    egen total_`x' = total(`x'), by(wave)
	egen max_total_`x' = max(total_`x')
	gen `x'_normalised = 0
	replace `x'_normalised = `x'/max_total_`x' if `x' > 0
}

gen weight_8 = 0 //all weights in single variable
replace weight_8 = wlrb_k_normalised if wave == 10
replace weight_8 = wlrf_o_normalised if wave == 14
replace weight_8 = wlrj_s_normalised if wave == 18


************************************
*** Create age-adjusted quantiles **
************************************

*note: this section takes some time to run but can be skipped because resultant variables are later merged 

gen L4_net_worth_quintile_age = .
local wealth_waves 6 10 14 18
forvalues z = 15/100{
	foreach x of local wealth_waves{
		capture noisily xtile L4_net_worth_`z'_quintile_`x' = L4.net_worth_e if wave == `x' & L2.hgage == `z' & weight_4 > 0 [pweight = weight_4], nq(5)
		capture noisily replace L4_net_worth_quintile_age = L4_net_worth_`z'_quintile_`x' if L2.hgage == `z' & weight_4 > 0 & wave == `x'
	}
}

gen L8_net_worth_quintile_age = .
local wealth_waves 10 14 18
forvalues z = 15/100{
	foreach x of local wealth_waves{
		capture noisily xtile L8_net_worth_`z'_quintile_`x' = L8.net_worth_e if wave == `x' & L6.hgage == `z' & weight_8 > 0 [pweight = weight_8], nq(5)
		capture noisily replace L8_net_worth_quintile_age = L8_net_worth_`z'_quintile_`x' if L6.hgage == `z' & weight_8 > 0 & wave == `x'
	}
}

gen L4_net_worth_tertile_age = .
local wealth_waves 6 10 14 18
forvalues z = 15/100{
	foreach x of local wealth_waves{
		capture noisily xtile L4_net_worth_`z'_tertile_`x' = L4.net_worth_e if wave == `x' & L2.hgage == `z' & weight_4 > 0 [pweight = weight_4], nq(3)
		capture noisily replace L4_net_worth_tertile_age = L4_net_worth_`z'_tertile_`x' if L2.hgage == `z' & weight_4 > 0 & wave == `x'
	}
}

gen L8_net_worth_tertile_age = .
local wealth_waves 10 14 18
forvalues z = 15/100{
	foreach x of local wealth_waves{
		capture noisily xtile L8_net_worth_`z'_tertile_`x' = L8.net_worth_e if wave == `x' & L6.hgage == `z' & weight_8 > 0 [pweight = weight_8], nq(3)
		capture noisily replace L8_net_worth_tertile_age = L8_net_worth_`z'_tertile_`x' if L6.hgage == `z' & weight_8 > 0 & wave == `x'
	}
}

gen L4_net_worth_half_age = .
local wealth_waves 6 10 14 18
forvalues z = 15/100{
	foreach x of local wealth_waves{
		capture noisily xtile L4_net_worth_`z'_half_`x' = L4.net_worth_e if wave == `x' & L2.hgage == `z' & weight_4 > 0 [pweight = weight_4], nq(2)
		capture noisily replace L4_net_worth_half_age = L4_net_worth_`z'_half_`x' if L2.hgage == `z' & weight_4 > 0 & wave == `x'
	}
}

gen L8_net_worth_half_age = .
local wealth_waves 10 14 18
forvalues z = 15/100{
	foreach x of local wealth_waves{
		capture noisily xtile L8_net_worth_`z'_half_`x' = L8.net_worth_e if wave == `x' & L6.hgage == `z' & weight_8 > 0 [pweight = weight_8], nq(2)
		capture noisily replace L8_net_worth_half_age = L8_net_worth_`z'_half_`x' if L6.hgage == `z' & weight_8 > 0 & wave == `x'
	}
}

***to run quickly, skip this section until here and merge the variables in

//keep xwaveid wave L4_net_worth_quintile_age L8_net_worth_quintile_age L4_net_worth_tertile_age L8_net_worth_tertile_age L4_net_worth_half_age L8_net_worth_half_age

//save net_worth_quantile_age.dta, replace

//merge 1:1 xwaveid wave using net_worth_quantile_age.dta
//drop _merge


************************************
*** Create standard quantiles ******
************************************

local wealth_waves 6 10 14 18
foreach x of local wealth_waves{
	xtile L4_net_worth_tertile_`x' = L4.net_worth_e if wave == `x' & weight_4 > 0 [pweight = weight_4], nq(3)
	replace L4_net_worth_tertile_`x' = 0 if L4_net_worth_tertile_`x' == . //this allows these to be summed later
}

gen L4_net_worth_tertile =  L4_net_worth_tertile_6 + L4_net_worth_tertile_10 + L4_net_worth_tertile_14 + L4_net_worth_tertile_18
replace L4_net_worth_tertile = . if L4_net_worth_tertile == 0

local wealth_waves 6 10 14 18
foreach x of local wealth_waves{
	xtile L4_net_worth_quintile_`x' = L4.net_worth_e if wave == `x' & weight_4 > 0 [pweight = weight_4], nq(5)
	replace L4_net_worth_quintile_`x' = 0 if L4_net_worth_quintile_`x' == .
}

gen L4_net_worth_quintile =  L4_net_worth_quintile_6 + L4_net_worth_quintile_10 + L4_net_worth_quintile_14 + L4_net_worth_quintile_18
replace L4_net_worth_quintile = . if L4_net_worth_quintile == 0

local wealth_waves 6 10 14 18
foreach x of local wealth_waves{
	xtile L4_net_worth_half_`x' = L4.net_worth_e if wave == `x' & weight_4 > 0 [pweight = weight_4], nq(2)
	replace L4_net_worth_half_`x' = 0 if L4_net_worth_half_`x' == .
}

gen L4_net_worth_half =  L4_net_worth_half_6 + L4_net_worth_half_10 + L4_net_worth_half_14 + L4_net_worth_half_18
replace L4_net_worth_half = . if L4_net_worth_half == 0

local wealth_waves 10 14 18
foreach x of local wealth_waves{
	xtile L8_net_worth_tertile_`x' = L8.net_worth_e if wave == `x' & weight_8 > 0 [pweight = weight_8], nq(3)
	replace L8_net_worth_tertile_`x' = 0 if L8_net_worth_tertile_`x' == .
}

gen L8_net_worth_tertile = L8_net_worth_tertile_10 + L8_net_worth_tertile_14 + L8_net_worth_tertile_18
replace L8_net_worth_tertile = . if L8_net_worth_tertile == 0

local wealth_waves 10 14 18
foreach x of local wealth_waves{
	xtile L8_net_worth_quintile_`x' = L8.net_worth_e if wave == `x' & weight_8 > 0 [pweight = weight_8], nq(5)
	replace L8_net_worth_quintile_`x' = 0 if L8_net_worth_quintile_`x' == .
}

gen L8_net_worth_quintile = L8_net_worth_quintile_10 + L8_net_worth_quintile_14 + L8_net_worth_quintile_18
replace L8_net_worth_quintile = . if L8_net_worth_quintile == 0

local wealth_waves 10 14 18
foreach x of local wealth_waves{
	xtile L8_net_worth_half_`x' = L8.net_worth_e if wave == `x' & weight_8 > 0 [pweight = weight_8], nq(2)
	replace L8_net_worth_half_`x' = 0 if L8_net_worth_half_`x' == .
}

gen L8_net_worth_half = L8_net_worth_half_10 + L8_net_worth_half_14 + L8_net_worth_half_18
replace L8_net_worth_half = . if L8_net_worth_half == 0


****************************************
*** Non-lifecycle adjusted models ******
****************************************

global controls_parent_chars_4 i.wave hgage hgage_squ i.sex L4.resident_children L4.partner i.L4.education i.mother_education i.father_education i.cob i.living_with_parents_at_14 siblings L4.HH_disposable_income 

global controls_parent_chars_8 i.wave hgage hgage_squ i.sex L8.resident_children L8.partner i.L8.education i.mother_education i.father_education i.cob i.living_with_parents_at_14 siblings L8.HH_disposable_income 


****Base specification models

*Real average effects (k=1) (chapter 2, in text)
reg delta_net_worth_4_real `controls_parent_chars_4' L2_0_bequest_real [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)

reg delta_net_worth_8_real `controls_parent_chars_8' L6_4_bequest_real [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)

*Average effects (k=1) (figure 2.7, table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_1

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_2

esttab model_1 model_2, ci(a3) keep(L2_0_bequest L6_4_bequest) scsv wide nostar noparentheses //figure 2.7

*k=2 (figure 2.9, table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_half [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_3

test 1.L4_net_worth_half#c.L2_0_bequest = 2.L4_net_worth_half#c.L2_0_bequest

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_half [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_4

test 1.L8_net_worth_half#c.L6_4_bequest = 2.L8_net_worth_half#c.L6_4_bequest

esttab model_3 model_4, ci(a3) keep(1.L4_net_worth_half#c.L2_0_bequest 2.L4_net_worth_half#c.L2_0_bequest  1.L8_net_worth_half#c.L6_4_bequest 2.L8_net_worth_half#c.L6_4_bequest) scsv wide nostar noparentheses //figure 2.9

*k=3 (table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_tertile [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_5

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_tertile [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_6

*k=5 (table B.10)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_quintile [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_7

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_quintile [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_8

esttab model_1 model_3 model_5 model_7 model_2 model_4 model_6 model_8, se(a3) keep(L2_0_bequest 1.L4_net_worth_half#c.L2_0_bequest 2.L4_net_worth_half#c.L2_0_bequest 1.L4_net_worth_tertile#c.L2_0_bequest 2.L4_net_worth_tertile#c.L2_0_bequest 3.L4_net_worth_tertile#c.L2_0_bequest 1.L4_net_worth_quintile#c.L2_0_bequest 2.L4_net_worth_quintile#c.L2_0_bequest 3.L4_net_worth_quintile#c.L2_0_bequest 4.L4_net_worth_quintile#c.L2_0_bequest 5.L4_net_worth_quintile#c.L2_0_bequest L6_4_bequest 1.L8_net_worth_half#c.L6_4_bequest 2.L8_net_worth_half#c.L6_4_bequest 1.L8_net_worth_tertile#c.L6_4_bequest 2.L8_net_worth_tertile#c.L6_4_bequest 3.L8_net_worth_tertile#c.L6_4_bequest 1.L8_net_worth_quintile#c.L6_4_bequest 2.L8_net_worth_quintile#c.L6_4_bequest 3.L8_net_worth_quintile#c.L6_4_bequest 4.L8_net_worth_quintile#c.L6_4_bequest 5.L8_net_worth_quintile#c.L6_4_bequest) scsv star (* 0.1 ** 0.05 *** 0.01)

****Models including people who did not inherit

*k=1 (table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest [pweight = weight_4] if weight_4 > 0 & wave != 18, vce(cluster xwaveid)
estimates store model_9

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_10

*k=2 (table B.9)

reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_half [pweight = weight_4] if weight_4 > 0 & wave != 18, vce(cluster xwaveid)
estimates store model_11

test 1.L4_net_worth_half#c.L2_0_bequest = 2.L4_net_worth_half#c.L2_0_bequest

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_half [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_12

test 1.L8_net_worth_half#c.L6_4_bequest = 2.L8_net_worth_half#c.L6_4_bequest

*k=3 (table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_tertile [pweight = weight_4] if weight_4 > 0 & wave != 18, vce(cluster xwaveid)
estimates store model_13

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_tertile [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_14

*k=5 (table B.10)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_quintile [pweight = weight_4] if wave != 18, vce(cluster xwaveid)
estimates store model_15

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_quintile [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_16

esttab model_9 model_11 model_13 model_15 model_10 model_12 model_14 model_16, se(a3) keep(L2_0_bequest 1.L4_net_worth_half#c.L2_0_bequest 2.L4_net_worth_half#c.L2_0_bequest 1.L4_net_worth_tertile#c.L2_0_bequest 2.L4_net_worth_tertile#c.L2_0_bequest 3.L4_net_worth_tertile#c.L2_0_bequest 1.L4_net_worth_quintile#c.L2_0_bequest 2.L4_net_worth_quintile#c.L2_0_bequest 3.L4_net_worth_quintile#c.L2_0_bequest 4.L4_net_worth_quintile#c.L2_0_bequest 5.L4_net_worth_quintile#c.L2_0_bequest L6_4_bequest 1.L8_net_worth_half#c.L6_4_bequest 2.L8_net_worth_half#c.L6_4_bequest 1.L8_net_worth_tertile#c.L6_4_bequest 2.L8_net_worth_tertile#c.L6_4_bequest 3.L8_net_worth_tertile#c.L6_4_bequest 1.L8_net_worth_quintile#c.L6_4_bequest 2.L8_net_worth_quintile#c.L6_4_bequest 3.L8_net_worth_quintile#c.L6_4_bequest 4.L8_net_worth_quintile#c.L6_4_bequest 5.L8_net_worth_quintile#c.L6_4_bequest) scsv star (* 0.1 ** 0.05 *** 0.01)


****0-3 year models including 2015-2018 inheritances

*k=1 (table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)
estimates store model_17

*k=2 (table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_half [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_18

test 1.L4_net_worth_half#c.L2_0_bequest = 2.L4_net_worth_half#c.L2_0_bequest

*k=3 (table B.9)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_tertile [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)
estimates store model_19

*k=5 (table B.10)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_quintile [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)
estimates store model_20

esttab model_17 model_18 model_19 model_20, se(a3) keep(L2_0_bequest 1.L4_net_worth_half#c.L2_0_bequest 2.L4_net_worth_half#c.L2_0_bequest 1.L4_net_worth_tertile#c.L2_0_bequest 2.L4_net_worth_tertile#c.L2_0_bequest 3.L4_net_worth_tertile#c.L2_0_bequest 1.L4_net_worth_quintile#c.L2_0_bequest 2.L4_net_worth_quintile#c.L2_0_bequest 3.L4_net_worth_quintile#c.L2_0_bequest 4.L4_net_worth_quintile#c.L2_0_bequest 5.L4_net_worth_quintile#c.L2_0_bequest) scsv star (* 0.1 ** 0.05 *** 0.01)



************************************
*** Lifecycle adjusted models ******
************************************

****Base specification models

*k=2 (table B.11)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_half_age [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_21

test 1.L4_net_worth_half_age#c.L2_0_bequest = 2.L4_net_worth_half_age#c.L2_0_bequest

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_half_age [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_22

test 1.L8_net_worth_half_age#c.L6_4_bequest = 2.L8_net_worth_half_age#c.L6_4_bequest

*k=3 (table B.11)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_tertile_age [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_23

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_tertile_age [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)
estimates store model_24

*k=5 (table B.12)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_quintile_age [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store model_25

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_quintile_age [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_26

esttab model_21 model_23 model_25 model_22 model_24 model_26, se(a3) keep(1.L4_net_worth_half_age#c.L2_0_bequest 2.L4_net_worth_half_age#c.L2_0_bequest 1.L4_net_worth_tertile_age#c.L2_0_bequest 2.L4_net_worth_tertile_age#c.L2_0_bequest 3.L4_net_worth_tertile_age#c.L2_0_bequest 1.L4_net_worth_quintile_age#c.L2_0_bequest 2.L4_net_worth_quintile_age#c.L2_0_bequest 3.L4_net_worth_quintile_age#c.L2_0_bequest 4.L4_net_worth_quintile_age#c.L2_0_bequest 5.L4_net_worth_quintile_age#c.L2_0_bequest 1.L8_net_worth_half_age#c.L6_4_bequest 2.L8_net_worth_half_age#c.L6_4_bequest 1.L8_net_worth_tertile_age#c.L6_4_bequest 2.L8_net_worth_tertile_age#c.L6_4_bequest 3.L8_net_worth_tertile_age#c.L6_4_bequest 1.L8_net_worth_quintile_age#c.L6_4_bequest 2.L8_net_worth_quintile_age#c.L6_4_bequest 3.L8_net_worth_quintile_age#c.L6_4_bequest 4.L8_net_worth_quintile_age#c.L6_4_bequest 5.L8_net_worth_quintile_age#c.L6_4_bequest) scsv star (* 0.1 ** 0.05 *** 0.01)


****Models including people who did not inherit

*k=2 (table B.11)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_half_age [pweight = weight_4] if weight_4 > 0 & wave != 18, vce(cluster xwaveid)
estimates store model_27

test 1.L4_net_worth_half_age#c.L2_0_bequest = 2.L4_net_worth_half_age#c.L2_0_bequest

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_half_age [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_28

test 1.L8_net_worth_half_age#c.L6_4_bequest = 2.L8_net_worth_half_age#c.L6_4_bequest

*k=3 (table B.11)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_tertile_age [pweight = weight_4] if weight_4 > 0 & wave != 18, vce(cluster xwaveid)
estimates store model_29

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_tertile_age [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_30

*k=5 (table B.12)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_quintile_age [pweight = weight_4] if wave != 18, vce(cluster xwaveid)
estimates store model_31

reg delta_net_worth_8 `controls_parent_chars_8' c.L6_4_bequest#L8_net_worth_quintile_age [pweight = weight_8] if weight_8 > 0, vce(cluster xwaveid)
estimates store model_32

esttab model_27 model_29 model_31 model_28 model_30 model_32, se(a3) keep(1.L4_net_worth_half_age#c.L2_0_bequest 2.L4_net_worth_half_age#c.L2_0_bequest 1.L4_net_worth_tertile_age#c.L2_0_bequest 2.L4_net_worth_tertile_age#c.L2_0_bequest 3.L4_net_worth_tertile_age#c.L2_0_bequest 1.L4_net_worth_quintile_age#c.L2_0_bequest 2.L4_net_worth_quintile_age#c.L2_0_bequest 3.L4_net_worth_quintile_age#c.L2_0_bequest 4.L4_net_worth_quintile_age#c.L2_0_bequest 5.L4_net_worth_quintile_age#c.L2_0_bequest 1.L8_net_worth_half_age#c.L6_4_bequest 2.L8_net_worth_half_age#c.L6_4_bequest 1.L8_net_worth_tertile_age#c.L6_4_bequest 2.L8_net_worth_tertile_age#c.L6_4_bequest 3.L8_net_worth_tertile_age#c.L6_4_bequest 1.L8_net_worth_quintile_age#c.L6_4_bequest 2.L8_net_worth_quintile_age#c.L6_4_bequest 3.L8_net_worth_quintile_age#c.L6_4_bequest 4.L8_net_worth_quintile_age#c.L6_4_bequest 5.L8_net_worth_quintile_age#c.L6_4_bequest) scsv star (* 0.1 ** 0.05 *** 0.01)


****0-3 year models including 2015-2018 inheritances

*k=2 (table B.11)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_half_age [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_33

test 1.L4_net_worth_half_age#c.L2_0_bequest = 2.L4_net_worth_half_age#c.L2_0_bequest

*k=3 (table B.11)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_tertile_age [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) , vce(cluster xwaveid)
estimates store model_34

*k=5 (table B.12)
reg delta_net_worth_4 `controls_parent_chars_4' c.L2_0_bequest#L4_net_worth_quintile_age [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)
estimates store model_35

esttab model_33 model_34 model_35, se(a3) keep(1.L4_net_worth_half_age#c.L2_0_bequest 2.L4_net_worth_half_age#c.L2_0_bequest 1.L4_net_worth_tertile_age#c.L2_0_bequest 2.L4_net_worth_tertile_age#c.L2_0_bequest 3.L4_net_worth_tertile_age#c.L2_0_bequest 1.L4_net_worth_quintile_age#c.L2_0_bequest 2.L4_net_worth_quintile_age#c.L2_0_bequest 3.L4_net_worth_quintile_age#c.L2_0_bequest 4.L4_net_worth_quintile_age#c.L2_0_bequest 5.L4_net_worth_quintile_age#c.L2_0_bequest) scsv star (* 0.1 ** 0.05 *** 0.01)



